/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import <Foundation/Foundation.h>

/** Notification fired when the connection status of the robot changes. */
#define CB_NOTIFICATION_CONNECTION_STATUS_CHANGED @"CBConnectionStatusChanged"
#define CB_NOTIFICATION_PAUSED_STATE_CHANGED @"CBPausedStateChanged"

@class CBArmPosition;
@class CBArmSpeed;
@class CBDofVector;
@class CBMotorParameters;
@protocol CBPath;

/** The main cardboard robot class.  Most functionality to control the robot
 *  can be accessed from here.
 */
@interface CBRobot : NSObject

/** Gets or sets the length of the first arm segment (from pivot to pivot),
 *  measured in centimeters.
 *
 *  You should generally only need to change this value if you replace the arm
 *  segment with a custom-sized one.
 */
@property (assign, nonatomic) double armLength1;

/** Gets or sets the length of the second arm segment (from pivot to pivot),
 *  measured in centimeters.
 *
 *  You should generally only need to change this value if you replace the arm
 *  segment with a custom-sized one.
 */
@property (assign, nonatomic) double armLength2;

/** Gets or sets the current position of the robot's arm.  This value is
 *  frequently overwritten with data read from the device, and setting it
 *  will generally have no effect.  Set #targetPosition and #speed
 *  instead to make the robot's arm move.
 */
@property (retain, nonatomic) CBArmPosition *currentPosition;

/** Gets or sets the target position of the robot's arm. */
@property (retain, nonatomic) CBArmPosition *targetPosition;

/** Gets or sets the desired speed of the robot's arm. */
@property (retain, nonatomic) CBArmSpeed *speed;

/** Gets the robot's default speed. */
@property (readonly, nonatomic) CBArmSpeed *defaultSpeed;

/** Gets whether or not the robot device is currently connected.  Connection to
 *  the device is automatic; there's no need to manually connect.
 */
@property (readonly, nonatomic) BOOL connected;

/** Gets or sets whether the robot is currently in a paused state. */
@property (assign, nonatomic) BOOL paused;

/** Specifies the path that the robot should execute.  Note that the target
 *  position and speed cannot be updated manually while a path is executing.
 *  Set this property to nil to have the robot stop executing the current path.
 */
@property (retain, nonatomic) id <CBPath> path;

- (id)init;

- (void)updateFromDevice;

- (CBMotorParameters *)motorParametersForMotor:(int)motorNumber;

- (double)angleFromSteps:(double)steps forMotor:(int)motorNumber;
- (double)stepsFromAngle:(double)angle forMotor:(int)motorNumber;

- (BOOL)isAtTargetPositionWithThreshold:(double)steps;
- (BOOL)isMotor:(int)motorNumber atTargetPositionWithThreshold:(double)steps;

- (void)setHomePosition;

@end
