/*=========================================================================
   This file is part of the Cardboard Robot SDK.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBStatusViewController.h"

static CIColor *runningBaseColor;
static CIColor *runningHighlightColor;
static CIColor *pausedBaseColor;
static CIColor *pausedHighlightColor;

/* Private members */
@interface CBStatusViewController ()

- (IBAction)toggleStatus:(id)sender;

- (void)updateButtonAndText;
- (void)setButtonColor1:(CIColor *)color1 andColor2:(CIColor *)color2;
- (void)clearButtonColor;

@end


@implementation CBStatusViewController

@synthesize delegate;
@synthesize connected;
@synthesize paused;

// (static initializer)
+ (void)initialize {
    if (runningBaseColor == nil) { runningBaseColor = [[CIColor colorWithRed:0.0 green:0.3 blue:0.03] retain]; }
    if (runningHighlightColor == nil) { runningHighlightColor = [[CIColor colorWithRed:0.0 green:1.0 blue:0.18] retain]; }
    if (pausedBaseColor == nil) { pausedBaseColor = [[CIColor colorWithRed:0.22 green:0.01 blue:0.0] retain]; }
    if (pausedHighlightColor == nil) { pausedHighlightColor = [[CIColor colorWithRed:1.0 green:0.14 blue:0.17] retain]; }
}

// (inherited from NSViewController)
- (id)initWithNibName:(NSString *)nibNameOrNil bundle:(NSBundle *)nibBundleOrNil {
    self = [super initWithNibName:nibNameOrNil bundle:nibBundleOrNil];
    if (self) {
        paused = YES;
        
        // Bind the outlets now instead of later
        [self loadView];
        
        [self updateButtonAndText];
    }
    return self;
}

// (property documented in header)
- (void)setConnected:(BOOL)value {
    if (connected != value) {
        connected = value;
        [self updateButtonAndText];
    }
}

// (property documented in header)
- (void)setPaused:(BOOL)value {
    if (paused != value) {
        paused = value;
        [self updateButtonAndText];
        
        if ([delegate respondsToSelector:@selector(statusViewPauseStateDidChange:)]) {
            [delegate statusViewPauseStateDidChange:self];
        }
    }
}

/** Called when the status button is clicked */
- (IBAction)toggleStatus:(id)sender {
    if ([self connected]) {
        [self setPaused:![self paused]];
    }
}

/** Updates the state of the button to match the current connected / running states */
- (void)updateButtonAndText {
    if (connected) {
        [button setEnabled:YES];
        [button setState:NSOnState];    // (false colors look better when in the "On" state)
        if (paused) {
            [self setButtonColor1:pausedBaseColor
                        andColor2:pausedHighlightColor];
            [largeText setStringValue:@"Paused"];
            [smallText setStringValue:@"Click to Start"];
        }
        else {
            [self setButtonColor1:runningBaseColor
                        andColor2:runningHighlightColor];
            [largeText setStringValue:@"Running"];
            [smallText setStringValue:@"Click to Stop"];
        }
    }
    else {
        [button setEnabled:NO];
        [button setState:NSOffState];
        [largeText setStringValue:@""]; // (let the placeholder text show)
        [smallText setStringValue:@""];
        [self clearButtonColor];
    }
}

/** Sets the colors of the button (color 1 = base color, color 2 = highlight) */
- (void)setButtonColor1:(CIColor *)color1 andColor2:(CIColor *)color2; {
    
    // Apply a "false color" filter to the button.  Note that the CALayer has
    // already been created in the XIB file.
    CALayer *layer = [button layer];
    CIFilter *filter = [CIFilter filterWithName:@"CIFalseColor"];
    [filter setDefaults];
    [filter setValue:color1 forKey:@"inputColor0"];
    [filter setValue:color2 forKey:@"inputColor1"];
    NSArray *filterArray = [NSArray arrayWithObject:filter];
    [layer setFilters:filterArray];
}

/** Clears the button's color */
- (void)clearButtonColor {
    [[button layer] setFilters:nil];
}

@end
