/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBJoystickManager.h"
#import "CBJoystickSettings.h"
#import "CBJoystickTranslator.h"

/* (Private members) */
@interface CBJoystickTranslator() {
    // Array of dictionaries, one for each motor, that map joystick
    // keys to positions.  Since we use only the sum, multiple joystick
    // elements can be used to control the same motors in a graceful way.
    NSArray *motorPositions;
    double sums[4];
    int counts[4];
}

- (void)processCommand:(NSString *)command withParameter:(double)parameter andElement:(NSString *)elementKey;
- (void)setPosition:(double)position forMotor:(NSUInteger)motorNumber andElement:(NSString *)elementKey;
- (void)updatePositionSums;

@end


@implementation CBJoystickTranslator

@synthesize enabled;

- (void)dealloc {
    [[CBJoystickManager sharedInstance] removeJoystickListener:self];
    [motorPositions release]; motorPositions = nil;
}

/** Initializes and returns a CBJoystickTranslator */
- (id)init {
    self = [super init];
    if (self) {
        [[CBJoystickManager sharedInstance] addJoystickListener:self];
        
        motorPositions = [[NSArray arrayWithObjects:
                           [NSMutableDictionary dictionary],
                           [NSMutableDictionary dictionary],
                           [NSMutableDictionary dictionary],
                           [NSMutableDictionary dictionary],
                           nil] retain];
    }
    return self;
}

/** Clears any cached inputs */
- (void)resetInputs {
    for (NSMutableDictionary *dict in motorPositions) {
        [dict removeAllObjects];
    }
    [self updatePositionSums];
}

/** Returns the position sum associated with the given motor (1 - 4) */
- (double)positionSumForMotor:(NSUInteger)motorNumber {
    double value = sums[motorNumber - 1];
    return MAX(-1.0, MIN(1.0, value));
}

/** Returns the number of active position sources for the given motor (1 - 4) */
- (int)sourceCountForMotor:(NSUInteger)motorNumber {
    return counts[motorNumber - 1];
}

/** Called when a joystick axis is changed */
- (void)joystick:(DDHidJoystick *)joystick
           stick:(unsigned)stick
            axis:(unsigned)axis
    valueChanged:(double)value {
    
    if (enabled) {
        
        // Take the dead zone into account, and scale.
        if (value > CB_JOYSTICK_DEAD_ZONE) {
            value = (value - CB_JOYSTICK_DEAD_ZONE) / (1.0 - CB_JOYSTICK_DEAD_ZONE);
        }
        else if (value < -CB_JOYSTICK_DEAD_ZONE) {
            value = -(-value - CB_JOYSTICK_DEAD_ZONE) / (1.0 - CB_JOYSTICK_DEAD_ZONE);
        }
        else {
            value = 0.0;
        }
        
        // Get the command associated with the stick / axis
        NSString *command = [[CBJoystickSettings sharedInstance] commandForStick:stick andAxis:axis ofJoystick:joystick];
        
        // Generate a unique key for this joystick element
        NSString *elementKey = [NSString stringWithFormat:@"%@/%@",
                                [[CBJoystickSettings sharedInstance] keyForJoystick:joystick],
                                [[CBJoystickSettings sharedInstance] keyForStick:stick andAxis:axis]];
        
        [self processCommand:command withParameter:value andElement:elementKey];
    }
}

/** Called when a joystick button is pressed */
- (void)joystick:(DDHidJoystick *)joystick
      buttonDown:(unsigned)buttonNumber {
    
    if (enabled) {
        
        // Get the command associated with the stick / axis
        NSString *command = [[CBJoystickSettings sharedInstance] commandForButton:buttonNumber ofJoystick:joystick];
        
        // Generate a unique key for this joystick element
        NSString *elementKey = [NSString stringWithFormat:@"%@/%@",
                                [[CBJoystickSettings sharedInstance] keyForJoystick:joystick],
                                [[CBJoystickSettings sharedInstance] keyForButton:buttonNumber]];
        
        [self processCommand:command withParameter:1.0 andElement:elementKey];
    }
}

/** Called when a joystick button is released */
- (void)joystick:(DDHidJoystick *)joystick
        buttonUp:(unsigned)buttonNumber {
    
    if (enabled) {
        
        // Get the command associated with the stick / axis
        NSString *command = [[CBJoystickSettings sharedInstance] commandForButton:buttonNumber ofJoystick:joystick];
        
        // Generate a unique key for this joystick element
        NSString *elementKey = [NSString stringWithFormat:@"%@/%@",
                                [[CBJoystickSettings sharedInstance] keyForJoystick:joystick],
                                [[CBJoystickSettings sharedInstance] keyForButton:buttonNumber]];
        
        [self processCommand:command withParameter:0.0 andElement:elementKey];
    }
}

/** Processes the given command */
- (void)processCommand:(NSString *)command withParameter:(double)parameter andElement:(NSString *)elementKey {
    if ([command isEqualToString:CB_JOYSTICK_COMMAND_CONTROL_M1]) {
        [self setPosition:parameter forMotor:1 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_CONTROL_M2]) {
        [self setPosition:parameter forMotor:2 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_CONTROL_M3]) {
        [self setPosition:parameter forMotor:3 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_CONTROL_M4]) {
        [self setPosition:parameter forMotor:4 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_INCREASE_M1]) {
        [self setPosition:parameter forMotor:1 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_DECREASE_M1]) {
        [self setPosition:-parameter forMotor:1 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_INCREASE_M2]) {
        [self setPosition:parameter forMotor:2 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_DECREASE_M2]) {
        [self setPosition:-parameter forMotor:2 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_INCREASE_M3]) {
        [self setPosition:parameter forMotor:3 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_DECREASE_M3]) {
        [self setPosition:-parameter forMotor:3 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_INCREASE_M4]) {
        [self setPosition:parameter forMotor:4 andElement:elementKey];
    }
    else if ([command isEqualToString:CB_JOYSTICK_COMMAND_DECREASE_M4]) {
        [self setPosition:-parameter forMotor:4 andElement:elementKey];
    }
}

/** Updates an individual joystick's contribution to the cumulative motor position */
- (void)setPosition:(double)position forMotor:(NSUInteger)motorNumber andElement:(NSString *)elementKey {

    NSMutableDictionary *dict = [motorPositions objectAtIndex:(motorNumber - 1)];
    
    // Only update the dictionary if (a) the value is non-zero or (b) the element already exists.
    // Otherwise, the slightest joystick input will override typed in settings.
    if ([dict objectForKey:elementKey] != nil || position != 0.0) {
        [dict setObject:[NSNumber numberWithDouble:position] forKey:elementKey];
        [self updatePositionSums];
    }
}

/** Updates the cumulative position for each motor by adding the components
 *  from each joystick.
 */
- (void)updatePositionSums {
    for (int motorIndex = 0; motorIndex < 4; motorIndex ++) {
        __block double sum = 0.0;
        __block int count = 0;
        NSDictionary *dict = [motorPositions objectAtIndex:motorIndex];
        [dict enumerateKeysAndObjectsUsingBlock:^(id key, id obj, BOOL *stop) {
            sum += [(NSNumber *)obj doubleValue];
            count ++;
        }];
        sums[motorIndex] = sum;
        counts[motorIndex] = count;
    }
}

@end
