/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBJoystickSettings.h"
#import "CBAppDelegate.h"


/* Private members */
@interface CBJoystickSettings() {
    NSMutableDictionary *settingsMap;   // Map from joystick key to dictionary of joystick settings
}

- (NSMutableDictionary *)settingsForJoystick:(DDHidJoystick *)joystick;
- (void)saveToDefaults;

@end


@implementation CBJoystickSettings

- (void)dealloc {
    [settingsMap release]; settingsMap = nil;
    [super dealloc];
}

/** Returns the shared joystick settings instance */
+ (CBJoystickSettings *)sharedInstance {
    static CBJoystickSettings *sharedInstance = nil;
    @synchronized(self) {
        if (sharedInstance == nil) {
            sharedInstance = [[CBJoystickSettings alloc] init];
        }
        return sharedInstance;
    }
}

/** Initializes and returns a CBJoystickSettings object */
- (id)init {
    self = [super init];
    if (self) {
        // Create a copy of the settings loaded from the application preferences
        settingsMap = [(NSDictionary *)[[NSUserDefaults standardUserDefaults] objectForKey:CB_DEFAULTS_KEY_JOYSTICK_SETTINGS] mutableCopy];
        NSAssert(settingsMap != nil, @"Failed to obtain joystick settings");
    }
    return self;
}

/** Sets the command for a particular joystick axis (can be nil) */
- (void)setCommand:(NSString *)command forStick:(NSUInteger)stickNumber andAxis:(NSUInteger)axisNumber ofJoystick:(DDHidJoystick *)joystick {
    NSMutableDictionary *settings = [self settingsForJoystick:joystick];
    NSString *key = [self keyForStick:stickNumber andAxis:axisNumber];
    if (command == nil) {
        [settings removeObjectForKey:key];
    }
    else {
        [settings setObject:command forKey:key];
    }
    [self saveToDefaults];
}

/** Sets the command for a particular joystick button (can be nil) */
- (void)setCommand:(NSString *)command forButton:(NSUInteger)buttonNumber ofJoystick:(DDHidJoystick *)joystick {
    NSMutableDictionary *settings = [self settingsForJoystick:joystick];
    NSString *key = [self keyForButton:buttonNumber];
    if (command == nil) {
        [settings removeObjectForKey:key];
    }
    else {
        [settings setObject:command forKey:key];
    }
    [self saveToDefaults];
}

/** Returns the command bound to a particular joystick axis, or nil if no command is bound */
- (NSString *)commandForStick:(NSUInteger)stickNumber andAxis:(NSUInteger)axisNumber ofJoystick:(DDHidJoystick *)joystick {
    NSMutableDictionary *settings = [self settingsForJoystick:joystick];
    NSString *key = [self keyForStick:stickNumber andAxis:axisNumber];
    return [settings objectForKey:key];
}

/** Returns the command bound to a particular joystick button, or nil if no command is bound */
- (NSString *)commandForButton:(NSUInteger)buttonNumber ofJoystick:(DDHidJoystick *)joystick {
    NSMutableDictionary *settings = [self settingsForJoystick:joystick];
    NSString *key = [self keyForButton:buttonNumber];
    return [settings objectForKey:key];
}

/** Returns a string that will be used to uniquely identify the given joystick
 *  in the persisted settings.  (NOTE: location ID is used so that more than
 *  one of the same type of joystick can be used, where each controls separate
 *  motors)
 */
- (NSString *)keyForJoystick:(DDHidJoystick *)joystick {
    return [NSString stringWithFormat:@"joystick-%u-%u-%u", [joystick vendorId],
            [joystick productId], [joystick locationId]];
}

/** Returns a string that uniquely identifies a particular axis on a joystick */
- (NSString *)keyForStick:(NSUInteger)stickNumber andAxis:(NSUInteger)axisNumber {
    return [NSString stringWithFormat:@"stick-%u-axis-%u", stickNumber, axisNumber];
}

/** Returns a string that uniquely identifies a particular button on a joystick */
- (NSString *)keyForButton:(NSUInteger)buttonNumber {
    return [NSString stringWithFormat:@"button-%u", buttonNumber];
}

/** Returns the settings dictionary for the given joystick */
- (NSMutableDictionary *)settingsForJoystick:(DDHidJoystick *)joystick {
    NSString *key = [self keyForJoystick:joystick];
    NSMutableDictionary *result = [settingsMap objectForKey:key];
    if (result == nil) {
        result = [NSMutableDictionary dictionary];
        [settingsMap setObject:result forKey:key];
    }
    return result;
}

/** Saves the current settings */
- (void)saveToDefaults {
    [[NSUserDefaults standardUserDefaults] setObject:settingsMap forKey:CB_DEFAULTS_KEY_JOYSTICK_SETTINGS];
}

@end
