/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBScalarSpeedTransformer.h"


/* Private members */
@interface CBScalarSpeedTransformer()

- (void)updateTransformedSpeed;
- (double)dofConversionFactor;

@end


@implementation CBScalarSpeedTransformer

@synthesize dofUnit;
@synthesize speed;
@synthesize transformedSpeed;

- (id)init {
    self = [super init];
    if (self) {
        dofUnit = CBUnitRadians;
    }
    return self;
}

+ (NSSet *)keyPathsForValuesAffectingSpeed {
    return [NSSet setWithObjects:@"transformedSpeed", nil];
}

+ (NSSet *)keyPathsForValuesAffectingTransformedSpeed {
    return [NSSet setWithObjects:@"speed", @"dofUnit", nil];
}

// (property documented in header)
- (void)setDofUnit:(CBUnit)aDofUnit {
    if (aDofUnit != dofUnit) {
        dofUnit = aDofUnit;
        [self updateTransformedSpeed];
    }
}

// (property documented in header)
- (void)setSpeed:(double)value {
    if (speed != value) {
        speed = value;
        [self updateTransformedSpeed];
    }
}

// (property documented in header)
- (void)setTransformedSpeed:(double)value {
    if (transformedSpeed != value) {
        transformedSpeed = value;
        speed = value / [self dofConversionFactor];
    }
}

/** Updates the transformed speed value */
- (void)updateTransformedSpeed {
    transformedSpeed = speed * [self dofConversionFactor];
}

/** Returns the conversion factor (radians to target unit) */
- (double)dofConversionFactor {
    switch (dofUnit) {
        case CBUnitRadians:
            return 1.0;
        case CBUnitDegrees:
            return 180.0 / pi;
        default:
            [NSException raise:@"CBInvalidOperation" format:@"Unknown unit: %i", dofUnit];
            return NAN;
    }
}

@end
