/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBCompiledProgram.h"
#import "CBJoystickSetupWindowController.h"
#import "CBJoystickTranslator.h"
#import "CBMainWindowController.h"
#import "CBProgram.h"
#import "CBProgramEntry.h"
#import "CBScalarSpeedTransformer.h"
#import "CBStatusViewController.h"

/** Update the current position at this interval (in seconds) */
#define CB_POSITION_UPDATE_INTERVAL     0.05


/* Private members */
@interface CBMainWindowController()

@property (retain, nonatomic) CBPositionTransformer *currentPositionTransformer;
@property (retain, nonatomic) CBPositionTransformer *targetPositionTransformer;
@property (retain, nonatomic) CBSpeedTransformer *speedTransformer;
@property (assign, nonatomic) BOOL controlsEnabled;

- (void)updateTimerElapsed:(NSTimer *)timer;
- (void)robotStatusDidChange:(NSNotification *)notification;
- (void)statusViewPauseStateDidChange:(id)sender;

- (void)updateStatusView;
- (void)installUpdateTimer;

- (IBAction)dofUnitDidChange:(id)sender;
- (IBAction)targetPositionChanged:(id)sender;
- (IBAction)speedChanged:(id)sender;
- (IBAction)setHomePosition:(id)sender;
- (IBAction)pressedProgramButton:(id)sender;
- (IBAction)pressedSaveOrLoad:(id)sender;
- (IBAction)pressedRunProgramButton:(id)sender;
- (IBAction)pressedJoystickSetupButton:(id)sender;
- (void)joystickSetupWindowWillClose:(NSNotification *)notification;

@end


@implementation CBMainWindowController

@synthesize robot;
@synthesize currentPositionTransformer;
@synthesize targetPositionTransformer;
@synthesize speedTransformer;
@synthesize controlsEnabled;

- (void)dealloc {

    [[NSNotificationCenter defaultCenter] removeObserver:self];
    [statusViewController setDelegate:nil];
    [robot removeObserver:self forKeyPath:@"targetPosition"];
    [robot removeObserver:self forKeyPath:@"speed"];
    [robot removeObserver:self forKeyPath:@"path"];
    
    [statusViewController release]; statusViewController = nil;
    [speedBeforeRunningProgram release]; speedBeforeRunningProgram = nil;
    [joystickSetupWindowController release]; joystickSetupWindowController = nil;
    [joystickTranslator release]; joystickTranslator = nil;
    
    [robot release]; robot = nil;
    NSAssert(updateTimer == nil, @"Timer should have been released when the window was closed");
    [currentPositionTransformer release]; currentPositionTransformer = nil;
    [targetPositionTransformer release]; targetPositionTransformer = nil;
    [speedTransformer release]; speedTransformer = nil;
    
    [super dealloc];
}

/** Returns the shared window controller instance */
+ (CBMainWindowController *)sharedInstance {
    static CBMainWindowController *sharedInstance = nil;
    @synchronized(self) {
        if (sharedInstance == nil) {
            sharedInstance = [[CBMainWindowController alloc] initWithWindowNibName:@"MainWindow"];
        }
        return sharedInstance;
    }
}

- (id)initWithWindowNibName:(NSString *)windowNibName {
    self = [super initWithWindowNibName:windowNibName];
    if (self) {
        // Create a robot instance.
        robot = [[CBRobot alloc] init];
        controlsEnabled = YES;
        speedBeforeRunningProgram = [[CBArmSpeed zero] retain];
        joystickTranslator = [[CBJoystickTranslator alloc] init];
    }
    return self;
}

/** Called when this object is loaded from the NIB */
- (void)awakeFromNib {

    // Install a timer to update the current position.
    [self installUpdateTimer];
    
    // Replace the status view placeholder with the real status view.
    statusViewController = [[CBStatusViewController alloc] initWithNibName:@"CBStatusView" bundle:nil];
    [statusViewController setDelegate:self];
    if (statusViewPlaceholder != nil && statusViewController != nil) {
        [[statusViewController view] setFrame:[statusViewPlaceholder frame]];
        [mainView replaceSubview:statusViewPlaceholder with:[statusViewController view]];
    }
    
    // Register for changes to the robot.
    NSNotificationCenter *nc = [NSNotificationCenter defaultCenter];
    [nc addObserver:self selector:@selector(robotStatusDidChange:)
               name:CB_NOTIFICATION_CONNECTION_STATUS_CHANGED
             object:robot];
        
    // Initialize misc. members
    [self setCurrentPositionTransformer:[[CBPositionTransformer alloc] init]];
    [self setTargetPositionTransformer:[[CBPositionTransformer alloc] init]];
    [self setSpeedTransformer:[[CBSpeedTransformer alloc] init]];
    [[self speedTransformer] setSpeed:[robot speed]];
    
    [self dofUnitDidChange:self];   // (set the initial unit)
    
    [robot addObserver:self forKeyPath:@"targetPosition" options:NSKeyValueObservingOptionNew context:nil];
    [robot addObserver:self forKeyPath:@"speed" options:NSKeyValueObservingOptionNew context:nil];
    [robot addObserver:self forKeyPath:@"path" options:NSKeyValueObservingOptionNew|NSKeyValueObservingOptionOld context:nil];
    
    // Update with the robot's current state.
    [self updateStatusView];
}

/** Called when the window is closed */
- (void)windowWillClose:(NSNotification *)notification {
    
    // Release the timer, so it's no longer referencing this object.
    [updateTimer invalidate]; [updateTimer release]; updateTimer = nil;
}

/** Called when the update timer elapses */
- (void)updateTimerElapsed:(NSTimer *)timer {
    
    // Update the robot's position from the device.
    [robot updateFromDevice];

    // Update the transformer values; the display will be updated
    // automatically using bindings.
    [currentPositionTransformer setPosition:[robot currentPosition]];
    
    // Update the target position and speed from joystick input
    if ([joystickTranslator enabled]) {
        CBArmSpeed *defaultSpeed = [robot defaultSpeed];
        CBDofVector *oldTip = [[[currentPositionTransformer position] tipPosition] pointAsDofVectorForRobot:robot];
        CBDofVector *newTip = [[[targetPositionTransformer position] tipPosition] pointAsDofVectorForRobot:robot];
        double oldM4 = [[currentPositionTransformer position] m4];
        double newM4 = [[targetPositionTransformer position] m4];
        CBArmSpeed *newSpeed = [speedTransformer speed];
        
        if ([joystickTranslator sourceCountForMotor:1] > 0) {
            double m1 = [defaultSpeed m1Speed] * [joystickTranslator positionSumForMotor:1];
            newTip = [newTip setComponentWithIndex:0 toValue:[oldTip m1] + m1];
            newSpeed = [newSpeed setComponentWithIndex:0 toValue:ABS(m1)];
        }
        if ([joystickTranslator sourceCountForMotor:2] > 0) {
            double m2 = [defaultSpeed m2Speed] * [joystickTranslator positionSumForMotor:2];
            newTip = [newTip setComponentWithIndex:1 toValue:[oldTip m2] + m2];
            newSpeed = [newSpeed setComponentWithIndex:1 toValue:ABS(m2)];
        }
        if ([joystickTranslator sourceCountForMotor:3] > 0) {
            double m3 = [defaultSpeed m3Speed] * [joystickTranslator positionSumForMotor:3];
            newTip = [newTip setComponentWithIndex:2 toValue:[oldTip m3] + m3];
            newSpeed = [newSpeed setComponentWithIndex:2 toValue:ABS(m3)];
        }
        if ([joystickTranslator sourceCountForMotor:4] > 0) {
            double m4 = [defaultSpeed m4Speed] * [joystickTranslator positionSumForMotor:4];
            newM4 = oldM4 + m4;
            newSpeed = [newSpeed setComponentWithIndex:3 toValue:ABS(m4)];
        }

        // Set the new position / speed.  Extrapolate by 1 second (already done).
        [robot setSpeed:newSpeed];
        [robot setTargetPosition:[CBArmPosition armPositionWithTipPosition:newTip andM4:newM4]];
    }
}

/** Called when the robot's connection status changes */
- (void)robotStatusDidChange:(NSNotification *)notification {
    [self updateStatusView];
}

/** Called when the status view's paused status is changed */
- (void)statusViewPauseStateDidChange:(id)sender {
    [robot setPaused:[statusViewController paused]];
}

/** Updates the status button to match the robot's current state */
- (void)updateStatusView {
    [statusViewController setConnected:[robot connected]];
    [statusViewController setPaused:[robot paused]];
}

/** Installs the update timer, if not already installed */
- (void)installUpdateTimer {
    if (updateTimer == nil) {
        updateTimer = [[NSTimer scheduledTimerWithTimeInterval:CB_POSITION_UPDATE_INTERVAL
                                                        target:self
                                                      selector:@selector(updateTimerElapsed:)
                                                      userInfo:nil
                                                       repeats:YES] retain];
    }
}

/** Called when one of the unit buttons is clicked */
- (IBAction)dofUnitDidChange:(id)sender {
    switch ([dofUnitControl selectedSegment]) {
        case 0:
            dofUnit = CBUnitDegrees;
            [speedNumberFormatter setPositiveFormat:@"#0.## deg/s"];
            break;
        case 1:
            dofUnit = CBUnitRadians;
            [speedNumberFormatter setPositiveFormat:@"#0.## rad/s"];
            break;
        case 2:
            dofUnit = CBUnitSteps;
            [speedNumberFormatter setPositiveFormat:@"#0.## deg/s"];    // (degrees are used instead of steps for speed of a program entry)
            break;
        default:
            dofUnit = CBUnitUnknown;
            NSAssert(NO, @"Unknown unit selection");
    }

    [currentPositionTransformer setDofUnit:dofUnit];
    [targetPositionTransformer setDofUnit:dofUnit];
    [speedTransformer setDofUnit:dofUnit];
    [[self document] setDofUnit:dofUnit];
}

/** Called when the user changes the target position via the UI */
- (IBAction)targetPositionChanged:(id)sender {
    [robot setTargetPosition:[targetPositionTransformer position]];
    
    // Reset the joystick inputs such that you once again have to move each
    // stick out of the dead zone in order to be counted.
    [joystickTranslator resetInputs];
}

/** Called when the user changes the speed via the UI */
- (IBAction)speedChanged:(id)sender {
    [robot setSpeed:[speedTransformer speed]];
    
    // Reset the joystick inputs such that you once again have to move each
    // stick out of the dead zone in order to be counted.
    [joystickTranslator resetInputs];
}

/** Called when the "Set Home Position" button is clicked */
- (IBAction)setHomePosition:(id)sender {
    [robot setHomePosition];
    [targetPositionTransformer setPosition:[CBArmPosition zero]];
}

/** Called when an observed value changes */
- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context {

    // Whenever a new program step is executed, update the selection
    if ([keyPath isEqualToString:@"lastExecutedEntryIndex"]) {
        [arrayController setSelectionIndex:[(CBCompiledProgram *)object lastExecutedEntryIndex]];
    }
    
    if (object == robot) {
        if ([keyPath isEqualToString:@"targetPosition"]) {
            [targetPositionTransformer setPosition:[robot targetPosition]];
        }
        if ([keyPath isEqualToString:@"speed"]) {
            [speedTransformer setSpeed:[robot speed]];
        }
        if ([keyPath isEqualToString:@"path"]) {
            
            // Register for notification when the currently executing step changes
            if ([change objectForKey:NSKeyValueChangeOldKey] != [NSNull null]) {
                CBCompiledProgram *oldProgram = [change objectForKey:NSKeyValueChangeOldKey];
                [oldProgram removeObserver:self forKeyPath:@"lastExecutedEntryIndex"];
            }
            if ([change objectForKey:NSKeyValueChangeNewKey] != [NSNull null]) {
                CBCompiledProgram *program = [change objectForKey:NSKeyValueChangeNewKey];
                [program addObserver:self forKeyPath:@"lastExecutedEntryIndex" options:NSKeyValueObservingOptionNew context:nil];
            }
            
            // Disable the position / speed controls if running a program
            [self setControlsEnabled:[robot path] == nil];
            [runProgramButton setTitle:([robot path] == nil ? @"Run Program" : @"Stop Program")];

            // When stopping a program, set the target position to the current
            // position, and the speed to whatever it was before the program
            // was started.
            if ([robot path] == nil) {
                NSAssert(speedBeforeRunningProgram != nil, nil);
                [robot updateFromDevice];   // (get the very latest position)
                [robot setTargetPosition:[robot currentPosition]];
                [robot setSpeed:speedBeforeRunningProgram];
                [arrayController setSelectionIndexes:nil];
            }
        }
    }
    
    // Forward position changes from the transformer to the robot
    if (object == targetPositionTransformer && keyPath == @"position") {
        [robot setTargetPosition:[targetPositionTransformer position]];
    }
    
    // Forward speed changes from the transformer to the robot
    if (object == speedTransformer && keyPath == @"speed") {
        [robot setSpeed:[speedTransformer speed]];
    }
}

/** Called when the active document is changed */
- (void)setDocument:(NSDocument *)document {
    NSDocument *oldDocument = [self document];
    if (document != oldDocument) {
        
        [super setDocument:document];
        [(CBProgram *)document setDofUnit:dofUnit];
        
        // Close the old document, so it can be re-opened again later.  If we
        // don't do this, attempting to open documents that have previously
        // been opened won't work.
        [oldDocument close];
        
        // Re-install the update timer.  It may have been destroyed if the
        // window was closed when the document was switched.
        [self installUpdateTimer];
    }
}

/** Called when the window loads */
- (void)windowDidLoad {
    [self windowDidResize:nil];
}

/** Called when the window size changes */
- (void)windowDidResize:(NSNotification *)notification {
    
    // Resize the program segmented control to match the width of the
    // program's scroll view.  Note that this doesn't actually extend the
    // last segment.
    NSRect frame = [programSegmentedControl frame];
    frame.size.width = [programScrollView frame].size.width;
    [programSegmentedControl setFrame:frame];
    
    // Resize the last segment in the control to fill the remaining space
    // (note: this only works if each segment has a fixed width)
    double remainingWidth = frame.size.width - 2;   // (-2 for the 1 pixel border on either end)
    for (int i = 0; i < [programSegmentedControl segmentCount] - 1; i ++) {
        remainingWidth -= [programSegmentedControl widthForSegment:i] + 1;  // (1 for the 1 pixel border after each segment)
    }
    [programSegmentedControl setWidth:remainingWidth forSegment:2];
}

- (NSString *)windowTitleForDocumentDisplayName:(NSString *)displayName {
    return [@"Cardboard Robot Console" stringByAppendingFormat:@": %@", displayName];
}

/** Called when a button is pressed on the segmented control under the program */
- (IBAction)pressedProgramButton:(id)sender {
    NSSegmentedControl *segmentedControl = (NSSegmentedControl *)sender;
    switch ([segmentedControl selectedSegment]) {
        case 0:     // (add)
        {
            // Insert a new object
            CBProgramEntry *entry = [[[CBProgramEntry alloc] init] autorelease];
            [[entry positionTransformer] setPosition:[robot currentPosition]];
            [[entry speedTransformer] setSpeed:10 * pi / 180.0];
            [arrayController addObject:entry];
            break;
        }
        case 1:     // (remove)
        {
            // Remove the selected objects
            [arrayController remove:nil];
            break;
        }
        default:
            NSAssert(NO, @"Unknown segment index");
    }
    
}

/** Called when the save or load button is pressed */
- (IBAction)pressedSaveOrLoad:(id)sender {
    NSSegmentedControl *segmentedControl = (NSSegmentedControl *)sender;
    switch ([segmentedControl selectedSegment]) {
        case 0:     // (save)
        {
            [[self document] saveDocumentAs:self];
            break;
        }
        case 1:     // (load)
        {
            [[NSDocumentController sharedDocumentController] openDocument:self];
            break;
        }
        default:
            NSAssert(NO, @"Unknown segment index");
    }
}

/** Called when the "Run Program" button is pressed */
- (IBAction)pressedRunProgramButton:(id)sender {
    
    if ([robot path] == nil) {
        // Select the first item while we move to the start position
        [arrayController setSelectionIndex:0];
        
        // Save the speed just before running the program
        [speedBeforeRunningProgram release];
        speedBeforeRunningProgram = [[robot speed] retain];
        
        // Compile and run the program
        CBCompiledProgram *compiledProgram = [[[CBCompiledProgram alloc] initWithProgram:[self document]] autorelease];
        [robot setPath:compiledProgram];
    }
    else {
        // Stop the currently running program
        [robot setPath:nil];
    }
}

/** Called when the "Joystick Setup" button is pressed */
- (IBAction)pressedJoystickSetupButton:(id)sender {
    // Disable the joystick translator so the robot's not moving while joysticks are being configured
    [joystickTranslator setEnabled:NO];
    
    if (joystickSetupWindowController == nil) {
        joystickSetupWindowController = [[CBJoystickSetupWindowController alloc] initWithWindowNibName:@"JoystickSetupWindow"];
        [[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(joystickSetupWindowWillClose:)
                                                     name:CB_NOTIFICATION_JOYSTICK_SETUP_WINDOW_WILL_CLOSE
                                                   object:joystickSetupWindowController];
    }
    [joystickSetupWindowController showWindow:self];
}

/** Called when the joystick setup window is closed */
- (void)joystickSetupWindowWillClose:(NSNotification *)notification {
    // Re-enable the joystick translator when the joystick setup window is closed
    [joystickTranslator resetInputs];   // (settings have changed; buttons may not control the motors they used to)
    [joystickTranslator setEnabled:YES];
}

@end
