/*=========================================================================
   This file is part of the Cardboard Robot Console application.

   Copyright (C) 2012 Ken Ihara.

   This program is free software: you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation, either version 3 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program.  If not, see <http://www.gnu.org/licenses/>.
=========================================================================*/

#import "CBProgram.h"
#import "CBProgramEntry.h"
#import "CBCompiledProgram.h"
#import "CBMainWindowController.h"
#import "CBPositionTransformer.h"
#import "CBScalarSpeedTransformer.h"

static void ComputeSpeedVectorAndTime(CBArmPosition *targetPosition, CBArmPosition *startPosition,
                                      double speed, CBRobot *robot, CBArmSpeed **speedVector,
                                      double *timeToReach);

/* Helper structure for CBCompiledProgram */
@interface CBCompiledProgramEntry : NSObject

@property (assign, nonatomic) double endTime;                   // End time of the entry
@property (assign, nonatomic) int index;                        // Index of the entry in the original program
@property (retain, nonatomic) CBArmPosition *targetPosition;    // Target position for this entry
@property (retain, nonatomic) CBArmSpeed *speed;                // Speed for this entry

@end


/* Helper structure for CBCompiledProgram */
@implementation CBCompiledProgramEntry

@synthesize endTime;
@synthesize index;
@synthesize targetPosition;
@synthesize speed;

- (void)dealloc {
    [targetPosition release]; targetPosition = nil;
    [speed release]; speed = nil;
}

@end


/* Private members */
@interface CBCompiledProgram() {
    NSMutableArray *entries;
}

@property (assign, nonatomic) NSUInteger lastExecutedEntryIndex;

- (CBCompiledProgramEntry *)entryAtTime:(double)time;

@end


@implementation CBCompiledProgram

@synthesize lastExecutedEntryIndex;

- (void)dealloc {
    [entries release]; entries = nil;
    [super dealloc];
}

- (id)init {
    [NSException raise:@"CBBadCall" format:@"Called init method without arguments"];
    return nil;
}

/** Initializes and returns a CBCompiledProgram object for the specified program */
- (id)initWithProgram:(CBProgram *)program {
    self = [super init];
    if (self) {
        
        CBRobot *robot = [[CBMainWindowController sharedInstance] robot];
        
        // Compile the program into an array of target positions / speeds
        entries = [[NSMutableArray alloc] init];
        CBArmPosition *lastPosition = nil;
        double endTime = 0.0;
        for (int index = 0; index < [[program entries] count]; index ++) {
            CBProgramEntry *originalEntry = [[program entries] objectAtIndex:index];

            // Figure out how long it will take to reach the target position,
            // and the per-motor speed that will cause each motor to arrive at
            // the same time
            CBArmPosition *position = [[originalEntry positionTransformer] position];
            if (lastPosition == nil) { lastPosition = position; }
            double speed = [[originalEntry speedTransformer] speed];
            CBArmSpeed *speedVector = nil;
            double timeToReach = 0.0;
            ComputeSpeedVectorAndTime(position, lastPosition, speed, robot, &speedVector, &timeToReach);
            endTime += timeToReach + [originalEntry pause];
            
            CBCompiledProgramEntry *entry = [[CBCompiledProgramEntry alloc] init];
            [entry setEndTime:endTime];
            [entry setIndex:index];
            [entry setTargetPosition:position];
            [entry setSpeed:speedVector];
            [entries addObject:entry];
            
            lastPosition = position;
        }
    }
    return self;
}

// (inherited from CBPath)
- (CBArmPosition *)startPosition {
    if ([entries count] > 0) {
        return [[entries objectAtIndex:0] targetPosition];
    }
    else {
        return [CBArmPosition zero];
    }
}

// (inherited from CBPath)
- (void)getTargetPosition:(CBArmPosition **)armPosition andSpeed:(CBArmSpeed **)armSpeed forTime:(double)time {
    CBCompiledProgramEntry *entry = [self entryAtTime:time];
    if (armPosition) { *armPosition = entry ? [entry targetPosition] : [CBArmPosition zero]; }
    if (armSpeed) { *armSpeed = entry ? [entry speed] : [CBArmSpeed zero]; }
}

// (inherited from CBPath)
- (double)pathLength {
    if ([entries count] == 0) {
        return 0.0;
    }
    else {
        return [(CBCompiledProgramEntry *)[entries objectAtIndex:[entries count] -1] endTime];
    }
}

// (inherited from CBPath)
- (void)setPathTime:(double)time {
    CBCompiledProgramEntry *entry = [self entryAtTime:time];
    [self setLastExecutedEntryIndex:[entry index]];
}

/** Returns the entry associated with the given time value.  Returns the
 *  first entry if the specified time is less than zero, or the last entry if
 *  the specified time is beyond the end of the path.
 */
- (CBCompiledProgramEntry *)entryAtTime:(double)time {
    if ([entries count] == 0) {
        return nil;
    }
    if (time < 0) {
        return [entries objectAtIndex:0];
    }
    // Return the first entry with an end time that's not already passed
    for (CBCompiledProgramEntry *entry in entries) {
        if ([entry endTime] - time > 0) {
            return entry;
        }
    }
    return [entries objectAtIndex:[entries count] - 1];
}

@end

/** Computes the amount of time that will be required to reach the given target
 *  position at the given "overall speed", and also returns the per-motor speed
 *  vector that should be used to have all motors arrive at the same time.
 */
static void ComputeSpeedVectorAndTime(CBArmPosition *targetPosition, CBArmPosition *startPosition,
                                      double speed, CBRobot *robot, CBArmSpeed **speedVector,
                                      double *timeToReach) {
    NSCAssert(targetPosition != nil, nil);
    NSCAssert(startPosition != nil, nil);
    NSCAssert(speed > 0, nil);
    NSCAssert(robot != nil, nil);
    NSCAssert(speedVector != nil, nil);
    NSCAssert(timeToReach != nil, nil);
    *speedVector = nil;
    
    // Compute how long it should take to reach the target position at the given
    // overall speed
    CBDofVector *targetTip = [[targetPosition tipPosition] pointAsDofVectorForRobot:robot];
    CBDofVector *startTip = [[startPosition tipPosition] pointAsDofVectorForRobot:robot];
    double dM1 = [targetTip m1] - [startTip m1];
    double dM2 = [targetTip m2] - [startTip m2];
    double dM3 = [targetTip m3] - [startTip m3];
    double dM4 = [targetPosition m4] - [startPosition m4];
    double magnitude = sqrt(dM1 * dM1 + dM2 * dM2 + dM3 * dM3 + dM4 * dM4);
    double time = magnitude / speed;

    if (time > 0) {
        *speedVector = [CBArmSpeed armSpeedWithM1Speed:ABS(dM1 / time)
                                            andM2Speed:ABS(dM2 / time)
                                            andM3Speed:ABS(dM3 / time)
                                            andM4Speed:ABS(dM4 / time)];
    }
    else {
        *speedVector = [CBArmSpeed zero];
    }
    *timeToReach = time;
}
