// Bike Power Meter

// 0810 add % battery used in welcome, added report selector
// 0809 bug in km/ah?
// 0808 wheel diameter in km
// 0806 calibrated current sensor, minor change to a/d amps
// 0804 ints -> floats, major change
// 0803 change battery charged detect to 2 volts, low bat alarm, bug shows 0km/ah when coasting???
// 0802 rework average km/ah
// 0801 do not average zero speed data
// 0720 remount wheel sensor to eliminate double tick -> software to match
// 0712 changes to wheel speed interrupt (timer 4 overflow)
// 0711 changes to lcd
// 0710 add trip energy efficiency, low bat alarmm, 

// 0701 AHr used reset changes
// 2019_0502 increase amp, volt filter
// 2019_0501 add total Ahr used since last charge, many changes
// 2019_0130 works

//                      +---------------------+
//            TXO, INT3 |[ ] 1  MICRO  VIN [ ]| Raw power in, 6V
//            RXI, INT2 |[ ] 0         GND [ ]| GND
//                      |[ ] GND       RST [ ]| Reset
//                      |[ ] GND       VCC [ ]| VCC
//             SDA, INT1|[ ] 2 PD1  PF4 21 [ ]| 
//      SCL, INT0, OC0B |[ ] 3 PD0  PF5 20 [ ]| 
//                      |[ ] 4 PD4  PF6 19 [ ]| 
//                OC3A  |[ ] 5 PC6  PF7 18 [ ]| 
//                OC4D  |[ ] 6 PD7  PB1 15 [ ]| PCINT1, SCK
//                 INT6 |[ ] 7 PE6  PB3 14 [ ]| PCINT3, MISO
//               PCINT4 |[ ] 8 PB4  PB2 16 [ ]| PCINT2, MOSI
//                 OC1A |[ ] 9 PB5  PB6 10 [ ]| OC1B
//                      +---------------------+


#include <EEPROM.h> 
#include <Wire.h> 
#include <LiquidCrystal_I2C.h>      // Arduino-LiquidCrystal-I2C-library-master
LiquidCrystal_I2C lcd(0x27,16,2);   // LCD address 0x27, 16 columns x 2 line display

int   const scopePin = 9;           // used for debugging
int   const magPin   = 7;           // wheel sensor
float const wheelDiameter = 26.0;   // inches, 82" circ, 82 i/s = 4.7mph, 40 mph = 704 i/s, -> 9 p/s, 1mph=17ips
float const numberOfCells = 14;     // number of parallel cells, e.g. 13 (48V) or 14 (52V)

// ------------------------- Periodic int from Timer 1 --------------------------------------------------
int     volatile  unsigned task;// task scheduler
boolean volatile        xps100; // volatile when shared with interrupt
boolean volatile        xps50;  // ISR flags used by task scheduler
boolean volatile        xps25;  // e.g. sets 25x/sec
boolean volatile        xps10;  // 10x/sec
boolean volatile        xps05;  // 5x/sec
boolean volatile        xps02;  // 2x/sec
boolean volatile        xps01;  // 1x/sec
ISR(TIMER1_OVF_vect) {
  task++;                       // interrupt is called 100x/sec, 120 uS execution time
  xps01   = ((task%100)==0);    // 1x/sec, sets every second
  xps02   = ((task%50)==0);     // 2x/sec, times are approximate
  xps05   = ((task%25)==0);     // 5x/sec
  xps10   = ((task%10)==0);     // 10x/sec
  xps25   = ((task%5)==0);      // 25x/sec
  xps50   = ((task%2)==0);      // 50x/sec
  xps100  = true;               // 100x/sec, 16 mS
}

// ------------------------- Periodic int from Timer 4-------------------------------
// wheel speed sensor debouncing, 100 uS/tick
// turns interrupt back on after debounce period
int unsigned volatile debouncer;// interrupt is off while timer is non-zero
boolean volatile magPinState;   // sensor state after debounce time
ISR(TIMER4_OVF_vect) {          // 100 uS
  if(debouncer) {               // if debounce period has not expired yet...
    debouncer--;                // decrement until 0, every 100 uS, else turn interrupt back on
    if(!debouncer) {            // if timeout has been reached...
      if(digitalRead(magPin)==LOW) {
        EICRB = B00110000;      // int on rising edge
        magPinState = LOW;
      }
      else {
        EICRB = B00100000;      // int on falling edge
        magPinState = HIGH;
      }
      EIMSK = B01000000;        // enable int 6
      EIFR  = B01000000;        // clear pending interrupts casued by ringing
    }
  }
}

// --------------------------  INT6 interrupt ------------------------------------
// if wheel sensor signal is two low going pulses, remount sensor perpendicular to magnet path
// tested with several sensors at distances from 0" to 1"
// remount sensor perpendicular to wheel -> clean single pulse at .5", edges bounce for 200 uS
// 1 rev/sec -> 4.7mph

volatile float          wheelPeriod;          // end - start = period
volatile float          distPerRev;           // km
volatile float          distTrip;             // total distance traveled, km
volatile float          rps;                  // wheel revolutions per second
volatile float          kph;                  // speed in kilometers/hour
volatile float const    K1 = 3600000.0;       // 50x60x1000
volatile unsigned long  startMillis;          // start of pulse, 4,294,967 seconds
volatile unsigned long  endMillis;            // end pulse time 
volatile unsigned int   stopDetect;           // shows 0 kph when stopped

ISR(INT6_vect){                               // 50 uS execution time  
  EIMSK = 0;                                  // turn off interrupt
  debouncer = 6;                              // x 100 uS
  if(magPinState==LOW) {                      // process on falling edge
    distTrip += distPerRev;                  // km, isr 1x/rev by magnet/sensor
    endMillis = millis();                     // millis() is unsigned long
    wheelPeriod = endMillis - startMillis;    // calc period
//    kph = (((3600000. * distPerRev)/wheelPeriod) + 2.*kph)/3.; // calc speed, average for 3 revolutions
    kph = K1 * distPerRev/wheelPeriod;       // calc speed
    if(kph>99.0) kph = 99.0;                  // 62mph
    startMillis = endMillis;                  // measured milliseconds
    stopDetect = 30;                          // in 100 mS task, kph=0 after countdown=0 
  }                 
//  digitalWrite(scopePin, !digitalRead(scopePin));// will show false triggers
}

// -------------------------------- low battery alarm --------------------------
boolean backlightOnFlag;          // state of lcd backlight
void lowBatAlarm() {              // flash LCD backlight
  if(backlightOnFlag) {     
    backlightOnFlag = false;      // if on -> turn off
    lcd.noBacklight();
  }
  else {
    backlightOnFlag = true;       // if off -> turn on
    lcd.backlight();
  }
}

// ------------------------------- Start --------------------------------------------
void setup() {
  Serial.begin(9600);             // start serial debugger
  pinMode(scopePin, OUTPUT);      // setup scope pin for debugging
  pinMode(magPin, INPUT_PULLUP);  // pullup on magnetic sensor pin
  lcd.begin();                    // start lcd routines
  welcome();                      // called before task scheduler starts (timers now disabled)
  
// --------------------- TIMER 1 16 bit, 10 mS PIR used by Tasker, ---------------
// TCCR1A COM1A1  COM1A0  COM1B1  COM1B0  -----   -----   WGM11   WGM10
// TCCR1B ICNC1   ICNS1   ------  WGM13   WGM12   CS12    CS11    CS10
// TIMSK1 ----    ----    ICIE1   ----    OCIE1C  OCIE1B  OCIE1A  TOIE1
   TCCR1A = B00000011;  // Mode 15, OCR1A is TOP
   TCCR1B = B00011010;  // clk/8=.5uS/tick
   OCR1A  = 20000;      // 20000 x .5uS = 10mS
   TIMSK1 = 00000001;	  // enable interrupt on overflow

// ---------------------- Timer 3 16 bit, free running, Wheel speed ---------------------
// TCCR3A COM3A1  COM3A0  COM3B1  COM3B0  COM1C1  COM1C0  WGM11   WGM10
// TCCR3B ICNC3   ICNS3   ----    WGM33   WGM32   CS32    CS31    CS30
  TCCR3A = B00000000;   // Mode 0, 16 bit, freerun
  TCCR3B = B00000101;   // clk/1024 (max)
 
// ---------------------- TIMER 4, 10 bit, 10 mS PIR, switch debounce -------------------
// TCCR4A COM4A1  COM4A0  COM4B1  COM4B0  FOC4A   FOC4B   PWM4A   PWM4B
// TCCR4B PWM4X   PSR4    DTPS41  DTPS40  CS43    CS43    CS41    CS40
// TCCR4C COM4A1S COM4A0S COMBB1S COM4B0S COM4D1  COM4D0  POC4D   PWM4D
// TCCR4D FPIE4   FPEN4   FPNC4   FPES4   FPAC4   FPF4    WGM41   WGM40
// TIMSK4 OCIE4D  OCIE4A  OCIE4B  ----    ----    TOIE4   ----    ----
  TCCR4A = B00000000;   // no compares
  TCCR4B = B00000011;   // prescaler 4 bits
  TCCR4C = B00000000;   // not used
  TCCR4D = B00000000;   // WGM = 0
  TIMSK4 = B00000100;   // interrupt on TOP
  OCR4C  = 197;         // TOP, set to 100uS rollover, 10 kHz PIR

//  -------------------------------INT6 ------------------------------------------------
// EICRB  ----    ----    ISC61   ISC60   ----    ----    ----    ----
// EIMSK  ----    INT6    ----    ----    INT3    INT2    INT1    INT0
  EICRB = B00100000;    // 00 low level, 01 change, 10 falling edge, 11 rising edge
  EIMSK = B01000000;    // OR INT6
}

// ------------------------------------ Main ---------------------------
byte reportNumber;
long loops;                   // main loop executes in 5 uS, >200k/s
boolean lowBatFlag;           // set if battery is below critical threshold
void loop() {                 // main loop
  loops++;                    // used to show number of loops/second, 212,200 x/sec
  if(xps100) {                // 100x/sec, 10 mS
    xps100 = false;           // nothing to do in this task time slot
  }
  if(xps50) {                 // 50x/sec, 20 mS
    xps50 = false;            // nothing to do in this task time slot
  }
  if(xps25) {                 // 25x/sec, 40 mS
    xps25 = false;            // reset task flag
    readData();               // read sensors, average data
  }
  if(xps10) {                 // 10x/sec, 100 mS
    xps10 = false;            // speed calc every mag sensor tick,
    if(stopDetect) {          // so if no ticks -> no speed data
      stopDetect--;           // decrement counter
      if(!stopDetect) kph = 0;// wheel is not rotating, kph=0
    }
  }    
  if(xps05) {                 // 5x/sec, 200 mS
    xps05 = false;            // nothing to do in this task time slot
  }   
  if(xps02) {                 // 2x/sec, 500 mS
    xps02 = false;            // reset task flag
    if(lowBatFlag) lowBatAlarm();// if battery is low -> alarm
  }
  if(xps01) {                 // 1x/sec
    xps01 = false;            // reset task flag
    calc();                   // calculate power usage
    if(reportNumber==1) report1();// report to monitor
    if(reportNumber==2) report2();
    if(reportNumber==3) report3();
    if(reportNumber==4) report4();
    if(reportNumber==5) report5();
    if(reportNumber==6) report6();
    loops = 0;                // reset loop counter
    LCD();                    // report to lcd
  }
  if(Serial.available()) operatorInterface();
}

// ------------------------- read A/D, average data -------------------------
// battery end voltage is 2.5V x 14 cells = 35V
// executes in less than 400 uS

int         adc3raw;                    // adc current
int         adc2;                       // A/D results from battery volt divider
int         adc3;                       // A/D from current sensor amplifier
int   const offsetBits  = 19;           // experimantal, use report1, number of bits req'd to offset to zero
float       Volts;                      // battery voltage
float       Amps;                       // current from battery
float const lowBatV     = 45.0;         // low battery threshold = 45 Volts, ~10%
float const VoltsPerBit = 0.0646;       // 65V = 1024 bits .0646
float const AmpsPerBit  = 0.0459;       // 45A = 1024bits  .044
void readData(){
  adc2 = analogRead(A2);                // volts
  adc3 = analogRead(A3) - offsetBits;   // amps 
  if(adc3<1) adc3=1;                    // limit so nothing less than zero, 1 bit-> 46 mA
  Volts = ((adc2*VoltsPerBit)+49.0*Volts)/50.0;
  Amps  = ((adc3*AmpsPerBit)+49.0*Amps)/50.0;
  if(Volts<lowBatV) lowBatFlag = true;  // set low battery flag
  else              lowBatFlag = false; // for alarm
}

// ------------------------- Process Data ------------------------------
// Dolphin 52V, 11.6AH, 39.2V cutoff, Samsung 29E 18650 Cells, 2.9AHr/cell
// 11.0AH @ 4A, .06 Ohms (cell) -> .21 Ohm (pack)
// 14 cells, 3.7V/Cell, 4P, .22 Ohms/cell->.77 Ohms total
// battery R measured in lab .20 Ohms @ 2.5A, .215 @ 9.42A
// %  10    19    27    36    45    55    73    81    90    100   % discharged @ 4A rate (4P)
// V  56    54.6  53.2  52.2  51.1  49.3  48.6  47.6  46.2  39.2  14S pack voltage

float AHrTrip;                        // Amp hours consumed since power on
float AHrFromBat;                     // total amphours used from battery
float AHrLast;                        // last AHr stored to eeprom
float battOhms = .230;                // battery resistance as measured in lab + sense resistor
float batOpenCir;                     // battery voltage corrected for internal resistance
float kmPerAHr;                       // kilometers/AHr
float aveKmPerAHr;                    // average since power on
float Watts;                          // Watts
void calc() {                         // calculate power usage
  AHrTrip += (Amps/3600.0);           // 60s/m x 60m/h, AHr consumed in last second
  aveKmPerAHr = distTrip/AHrTrip;     // average
  Amps  = max(Amps,.1);               // make sure no divide by zero
  kmPerAHr = kph / Amps;              // same as km/AHr, instantaneous value
  batOpenCir = Volts + battOhms*Amps; // open circuit battery Volts corrected for internal IR drop
  if((AHrTrip-AHrLast) > .1) {        // 100mAHr has been used, update eeprom
    AHrFromBat = AHrFromBat+(AHrTrip-AHrLast);// running total of mAHr used from battery since chrg
    AHrLast = AHrTrip;                // zero counter
    EEPROM.put(10, AHrFromBat);       // accumulate total used from battery to eeprom
    EEPROM.put(20, batOpenCir);       // write last battery voltage corrected for resistance
  }
}

// -------------------------- Estimated % Discharged ---------------------------------
// cell voltage and % discharge data from Samsung data sheet
float batUsed;                    // % discharged
void calcBatUsed() {
//                0     1     2     3     4     5     6     7     8     9     10    11    12
float V[13] =  {4.16, 4.05, 3.95, 3.87, 3.79, 3.70, 3.62, 3.57, 3.53, 3.48, 3.38, 3.27, 2.80};
float pc[13] = {0.00, 8.68, 17.4, 26.0, 34.7, 43.4, 52.1, 60.8, 69.4, 78.1, 86.8, 95.5, 100.0};
float deltaPc;                    // delta % from i to i+1
float deltaV;                     // delta V from i to i+1
float pcPerV;                     // %/Volt
float delta;                      // 
float voltsPerCell;               // measured volts/cell

  int i = 0;
  voltsPerCell = Volts/numberOfCells;// volts/cell
  while(V[i+1]>voltsPerCell) i++;   // find array entry point
  deltaPc = pc[i+1] - pc[i];        // calc delta %
  deltaV = V[i] - V[i+1];           // calc delta V
  pcPerV = deltaPc/deltaV;          // %/V
  delta = pcPerV * (V[i] - voltsPerCell);// change
  batUsed = pc[i] + delta;          // cell % discharged
  if(voltsPerCell>4.15) batUsed = 0;
  if(voltsPerCell<2.8) batUsed = 100.;
}

// ----------------------------- Welcome -------------------------------------
void welcome() {                    // task scheduler is not running during welcome screens
//  EEPROM.put(10, AHrFromBat);     // uncomment these two lines to initialize eeprom for first time
//  EEPROM.put(20, batOpenCir);     // uncomment these two lines to initialize eeprom for first time
  distPerRev = 3.1415 * wheelDiameter * .0000254; // dist in km per wheel revolution
  startMillis = 1000;               // arbitrary non-zero number so first calc is valid
  wheelPeriod = 10000000;           // this is basically forever - bike is stopped when power on
  lcd.backlight();                  // turn on backlight
  backlightOnFlag = true;           // flag stores state of backlight
  EEPROM.get(10, AHrFromBat);       // get eeprom data from last ride, ahr
  EEPROM.get(20, batOpenCir);       // volts
  
  lcd.clear();                      // clear lcd
  lcd.print(AHrFromBat,1);          // show capacity used from battery so far
  lcd.print(" AHr used ");          //
  lcd.setCursor(0,1);               // second line
  lcd.print("from Battery");        //
  for(int i=0; i<1000; i++) {       // 5 seconds
    delay(5);                       // get data averages
    readData();                     // filters settled
  }

  lcd.clear();                      // clear
  calcBatUsed();                    // % discharged
  lcd.print("Est ");                //
  lcd.print(batUsed,0);             //
  lcd.print("% used");              //
  lcd.setCursor(0,1);               // second line
  lcd.print("based on Volts");      //
  delay(5000);                      // display time
  
  lcd.clear();                      // clear lcd
  lcd.print("Now      Last");        // titles
  lcd.setCursor(0,1);               // second line
  lcd.print(Volts,1);               // voltage now
  lcd.print(" V");                  // label
  lcd.setCursor(9,1);               // move cursor
  lcd.print(batOpenCir,1);          // show current battery voltage
  lcd.print(" V");                  // label
  delay(5000);                      // time to view
  if(Volts>(batOpenCir+2)) {        // if volts now > last volts saved -> bat has been charged
    AHrFromBat = 0;                 // newly charged battery detected
    EEPROM.put(0, AHrFromBat);      // store zero ah used in eeprom
    lcd.clear();                    // zero AHr used from battery
    lcd.print("Recharge detect ");  //
    lcd.setCursor(0,1);             // second line
    lcd.print("AHr used RESET  ");  // announcement
    delay(5000);                    // time to view
  }
}

// ------------------------------ LCD Report ------------------------------------------
void LCD() {
  lcd.clear();                // clear and home
  if(kmPerAHr>3.0) lcd.print(kmPerAHr,0);// show efficiency, float, only whole number
  else             lcd.print(kmPerAHr,1);
  lcd.print(" km/AH ");       // show current efficiency
  
  lcd.setCursor(10,0);         // column, line
  lcd.print(aveKmPerAHr,0);   // show average efficiency on this trip, float, only whole number
  lcd.print(" Ave ");         // space after label to erase leftovers of big number

  lcd.setCursor(0,1);         // 2nd line
  lcd.print(AHrFromBat,1);    // float, one decimal point
  lcd.print(" AHr ");         // show ah used from battery since last charge

  lcd.setCursor(9,1);         // col, line
  lcd.print(Amps,0);          // show current current level, float, no decimal data
  lcd.print("A ");            // filtered
  
  lcd.setCursor(13,1);        // show bat voltage
  lcd.print(batOpenCir,0);    // corrected for IR drop, float no decimal data
  lcd.print("V ");            // filtered
}


// ------------------------------ Report --------------------------------
char TAB = 9;
int lineCounter;

void operatorInterface() {
  char inNum;
  inNum = Serial.read();
  if(inNum=='0') reportNumber = 0;
  if(inNum=='1') reportNumber = 1;
  if(inNum=='2') reportNumber = 2;
  if(inNum=='3') reportNumber = 3;
  if(inNum=='4') reportNumber = 4;
  if(inNum=='5') reportNumber = 5;
  if(inNum=='6') reportNumber = 6;
  lineCounter = 0;
}

void report1() {
  if((lineCounter%20) == 0) Serial.println("Volts   Amps    adc2(V)  A3(A)   adc3");
  lineCounter++;
  Serial.print(Volts);
  Serial.print(TAB);
  Serial.print(Amps);
  Serial.print(TAB);
  Serial.print(adc2);
  Serial.print(TAB);
  Serial.print(adc3raw);
  Serial.print(TAB);
  Serial.print(adc3);
  Serial.println();
}

void report2() {
  if((lineCounter%20) == 0) Serial.println("Period  Speed");
  lineCounter++;
  Serial.print(wheelPeriod);
  Serial.print(TAB);
  Serial.print(kph);
  Serial.println();
}

void report3() {
  if((lineCounter%20) == 0) Serial.println("Speed   Volts   Amps    km/AHr  AHr     Loops");
  lineCounter++;
  Serial.print(kph);
  Serial.print(TAB);
  Serial.print(Volts);
  Serial.print(TAB);
  Serial.print(Amps);
  Serial.print(TAB);
  Serial.print(kmPerAHr);
  Serial.print(TAB);
  Serial.print(AHrTrip);
  Serial.print(TAB);
  Serial.print(loops);
  Serial.println();
}

void report4() {
  if((lineCounter%20) == 0) Serial.println("Speed   Volts   BatOC   Amps    AHr     km/AHr  BatAH   Ave");
  lineCounter++;
  Serial.print(kph);
  Serial.print(TAB);
  Serial.print(Volts,1);
  Serial.print(TAB);
  Serial.print(batOpenCir);
  Serial.print(TAB);
  Serial.print(Amps,1);
  Serial.print(TAB);
  Serial.print(AHrTrip,1);
  Serial.print(TAB);
  Serial.print(kmPerAHr);
  Serial.print(TAB);
  Serial.print(AHrFromBat);
  Serial.print(TAB);
  Serial.print(aveKmPerAHr);
  Serial.println();
}

void report5() {
  if((lineCounter%20) == 0) Serial.println("kph     AHr     km/AHr  Dist    Ave");
  lineCounter++;
  Serial.print(kph);
  Serial.print(TAB);
  Serial.print(AHrTrip);
  Serial.print(TAB);
  Serial.print(kmPerAHr);
  Serial.print(TAB);
  Serial.print(distTrip);
  Serial.print(TAB);
  Serial.print(aveKmPerAHr);
  Serial.println();
}

void report6() {
  if((lineCounter%20) == 0) Serial.println("Km/Hr   Amps    km/AHr  AHr");
  lineCounter++;
  Serial.print(kph);
  Serial.print(TAB);
  Serial.print(Amps);
  Serial.print(TAB);
  Serial.print(kmPerAHr);
  Serial.print(TAB);
  Serial.print(AHrTrip);
  Serial.println();
}
