/*
* Project: HID-usb48g for emu48
* Author: Michael Pabst (epostkastl@gmail.com)
* Creation Date (build): 2019-04-10

* Parts of this project are from:
* Project: HID-Test
* Author: Christian Starkjohann
* Creation Date: 2006-02-02
* Tabsize: 4
* Copyright: (c) 2006 by OBJECTIVE DEVELOPMENT Software GmbH
* License: GNU GPL v2 (see License.txt) or proprietary (CommercialLicense.txt)
* This Revision: $Id$
*/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <avr/pgmspace.h>
#include <avr/wdt.h>
#include <avr/delay.h>

#include "usbdrv.h"
#include "oddebug.h"

/* ----------------------- hardware I/O abstraction ------------------------ */

/* pin assignments:
PB0	A5
PB1	A4
PB2	A3
PB3	A2
PB4	A1
PB5 A0

PC0	A12
PC1	A13
PC2	A14
PC3	A15
PC4	A16
PC5	AR17

PD0	USB-
PD1	debug tx
PD2	USB+ (int0)
PD3	
PD4	A9
PD5	A10
PD6	A11
PD7	ON-KEY
*/

static void hardwareInit(void)
{
	uchar	i, j;

	PORTB = 0xff;   /* activate all pull-ups */
	DDRB = 0;       /* all pins input */
	PORTC = 0xff;   /* activate all pull-ups */
	DDRC = 0;       /* all pins input */
	PORTD = 0xfa;   /* 1111 1010 bin: activate pull-ups except on USB lines */
	DDRD = 0x07;    /* 0000 0111 bin: all pins input except USB (-> USB reset) */
	j = 0;
	while(--j){     /* USB Reset by device only required on Watchdog Reset */
		i = 0;
		while(--i); /* delay >10ms for USB reset */
	}
	DDRD = 0x02;    /* 0000 0010 bin: remove USB reset condition */
	/* configure timer 0 for a rate of 12M/(1024 * 256) = 45.78 Hz (~22ms) */
	TCCR0 = 5;      /* timer 0 prescaler: 1024 */
	DDRD |= (1<<4); // set bit -> Output
	DDRD |= (1<<5); // set bit -> Output
	DDRD |= (1<<6); // set bit -> Output

}

/* ------------------------------------------------------------------------- */

#define NUM_KEYS    49

/* The following function returns an index for the first key pressed. It
* returns 0 if no key is pressed.
*/
static uchar    keyPressed(void)
{
	uchar x = 0;

	if ((PIND & (1 << 7)) == 0) x = 49; //ON-KEY

	PORTC = 0b00000000;   
	DDRC  = 0b00100000; // AR17: 5keys
	_delay_ms(1);	   
	if ((PINB & (1 << 1)) == 0) x = 2;   //hp48-B
	if ((PINB & (1 << 2)) == 0) x = 3;   //hp48-C
	if ((PINB & (1 << 3)) == 0) x = 4;   //hp48-D
	if ((PINB & (1 << 4)) == 0) x = 5;   //hp48-E
	if ((PINB & (1 << 5)) == 0) x = 6;   //hp48-F

	DDRC  = 0b00010000; // A16: 5keys
	_delay_ms(1);
	if ((PINB & (1 << 1)) == 0) x = 8;   //hp48-H
	if ((PINB & (1 << 2)) == 0) x = 9;   //hp48-I
	if ((PINB & (1 << 3)) == 0) x = 10;  //hp48-J
	if ((PINB & (1 << 4)) == 0) x = 11;  //hp48-K
	if ((PINB & (1 << 5)) == 0) x = 12;  //hp48-L

	DDRC  = 0b00001000; // A15: 5keys
	_delay_ms(1);
	if ((PINB & (1 << 1)) == 0) x = 14;  //hp48-N
	if ((PINB & (1 << 2)) == 0) x = 15;  //hp48-O
	if ((PINB & (1 << 3)) == 0) x = 16;  //hp48-P 
	if ((PINB & (1 << 4)) == 0) x = 17;  //hp48-Q
	if ((PINB & (1 << 5)) == 0) x = 18;  //hp48-R

	DDRC  = 0b00000100; // A14: 5keys
	_delay_ms(1);
	if ((PINB & (1 << 1)) == 0) x = 20;  //hp48-T
	if ((PINB & (1 << 2)) == 0) x = 21;  //hp48-U
	if ((PINB & (1 << 3)) == 0) x = 22;  //hp48-V
	if ((PINB & (1 << 4)) == 0) x = 23;  //hp48-W
	if ((PINB & (1 << 5)) == 0) x = 24;  //hp48-X

	DDRC  = 0b00000010; // A13: 5keys
	_delay_ms(1);
	if ((PINB & (1 << 1)) == 0) x = 27;  //hp48-ENTER
	if ((PINB & (1 << 2)) == 0) x = 25;  //hp48-Y
	if ((PINB & (1 << 3)) == 0) x = 26;  //hp48-Z
	if ((PINB & (1 << 4)) == 0) x = 28;  //hp48-DELETE
	if ((PINB & (1 << 5)) == 0) x = 29;  //hp48-BACKSPACE

	DDRC  = 0b00000001; // A12: 6keys
	_delay_ms(1);
	if ((PINB & (1 << 0)) == 0) x = 44;	 //hp48-alpha
	if ((PINB & (1 << 1)) == 0) x = 19;	 //hp48-S
	if ((PINB & (1 << 2)) == 0) x = 37;	 //hp48-7
	if ((PINB & (1 << 3)) == 0) x = 38;	 //hp48-8
	if ((PINB & (1 << 4)) == 0) x = 39;  //hp48-9
	if ((PINB & (1 << 5)) == 0) x = 45;	 //hp48-DIV

	DDRC  = 0b00000000; 

	PORTD &= ~(1<<4);  // clear bit -> LOW  // A9: 5keys
	_delay_ms(1);
	if ((PINB & (1 << 1)) == 0) x = 13;  //hp48-M
	if ((PINB & (1 << 2)) == 0) x = 30;  //hp48-0 (zero)
	if ((PINB & (1 << 3)) == 0) x = 40;  //hp48-DOT
	if ((PINB & (1 << 4)) == 0) x = 41;  //hp48-SPAC
	if ((PINB & (1 << 5)) == 0) x = 48;  //hp48-PLUS
	PORTD |= (1<<4); // set high

	PORTD &= ~(1<<5);  // clear bit -> LOW  // A10: 6 keys
	_delay_ms(1);
	if ((PINB & (1 << 0)) == 0) x = 43;  //hp-RS
	if ((PINB & (1 << 1)) == 0) x = 1;   //hp48-A
	if ((PINB & (1 << 2)) == 0) x = 31;	 //hp48-1
	if ((PINB & (1 << 3)) == 0) x = 32;  //hp48-2
	if ((PINB & (1 << 4)) == 0) x = 33;	 //hp48-3
	if ((PINB & (1 << 5)) == 0) x = 47;	 //hp48-MINUS
	PORTD |= (1<<5); // set high

	PORTD &= ~(1<<6);  // clear bit -> LOW  // A11: 6 keys
	_delay_ms(1);
	if ((PINB & (1 << 0)) == 0) x = 42;  //hp48-LS
	if ((PINB & (1 << 1)) == 0) x = 7; 	 //hp48-G
	if ((PINB & (1 << 2)) == 0) x = 34;  //hp48-4
	if ((PINB & (1 << 3)) == 0) x = 35;  //hp48-5
	if ((PINB & (1 << 4)) == 0) x = 36;  //hp48-6
	if ((PINB & (1 << 5)) == 0) x = 46;  //hp48-MUL
	PORTD |= (1<<6); // set high

	if (x>0) return x;
	return 0;
}


/* ------------------------------------------------------------------------- */
/* ----------------------------- USB interface ----------------------------- */
/* ------------------------------------------------------------------------- */

static uchar    reportBuffer[2];    /* buffer for HID reports */
static uchar    idleRate;           /* in 4 ms units */

const PROGMEM char usbHidReportDescriptor[35] = {   /* USB report descriptor */
	0x05, 0x01,                    // USAGE_PAGE (Generic Desktop)
	0x09, 0x06,                    // USAGE (Keyboard)
	0xa1, 0x01,                    // COLLECTION (Application)
	0x05, 0x07,                    //   USAGE_PAGE (Keyboard)
	0x19, 0xe0,                    //   USAGE_MINIMUM (Keyboard LeftControl)
	0x29, 0xe7,                    //   USAGE_MAXIMUM (Keyboard Right GUI)
	0x15, 0x00,                    //   LOGICAL_MINIMUM (0)
	0x25, 0x01,                    //   LOGICAL_MAXIMUM (1)
	0x75, 0x01,                    //   REPORT_SIZE (1)
	0x95, 0x08,                    //   REPORT_COUNT (8)
	0x81, 0x02,                    //   INPUT (Data,Var,Abs)
	0x95, 0x01,                    //   REPORT_COUNT (1)
	0x75, 0x08,                    //   REPORT_SIZE (8)
	0x25, 0x65,                    //   LOGICAL_MAXIMUM (101)
	0x19, 0x00,                    //   USAGE_MINIMUM (Reserved (no event indicated))
	0x29, 0x65,                    //   USAGE_MAXIMUM (Keyboard Application)
	0x81, 0x00,                    //   INPUT (Data,Ary,Abs)
	0xc0                           // END_COLLECTION
};
/* We use a simplifed keyboard report descriptor which does not support the
* boot protocol. We don't allow setting status LEDs and we only allow one
* simultaneous key press (except modifiers). We can therefore use short
* 2 byte input reports.
* The report descriptor has been created with usb.org's "HID Descriptor Tool"
* which can be downloaded from http://www.usb.org/developers/hidpage/.
* Redundant entries (such as LOGICAL_MINIMUM and USAGE_PAGE) have been omitted
* for the second INPUT item.
*/

/* Keyboard usage values, see usb.org's HID-usage-tables document, chapter
* 10 Keyboard/Keypad Page for more codes.
*/
#define KEY_RS              (1<<0) //MOD_CONTROL_LEFT    
#define KEY_LS				(1<<1) //MOD_SHIFT_LEFT
#define MOD_ALT_LEFT        (1<<2)
#define MOD_GUI_LEFT        (1<<3)
#define MOD_CONTROL_RIGHT   (1<<4)
#define MOD_SHIFT_RIGHT     (1<<5)
#define MOD_ALT_RIGHT       (1<<6)
#define MOD_GUI_RIGHT       (1<<7)

#define KEY_A       4
#define KEY_B       5
#define KEY_C       6
#define KEY_D       7
#define KEY_E       8
#define KEY_F       9
#define KEY_G       10
#define KEY_H       11
#define KEY_I       12
#define KEY_J       13
#define KEY_K       14
#define KEY_L       15
#define KEY_M       16
#define KEY_N       17
#define KEY_O       18
#define KEY_P       19
#define KEY_Q       20
#define KEY_R       21
#define KEY_S       22
#define KEY_T       23
#define KEY_U       24
#define KEY_V       25
#define KEY_W       26
#define KEY_X       27
#define KEY_Y       28
#define KEY_Z       29

#define KEY_1       30
#define KEY_2       31
#define KEY_3       32
#define KEY_4       33
#define KEY_5       34
#define KEY_6       35
#define KEY_7       36
#define KEY_8       37
#define KEY_9       38
#define KEY_0       39
#define KEY_ENTER   40
#define KEY_ON     41 //ESC

#define KEY_BACKSPACE 42
#define KEY_DOT     55

#define KEY_SPACE   44
#define KEY_ALPHA   43
#define KEY_DIV     84
#define KEY_MUL     85
#define KEY_MINUS   86
#define KEY_PLUS    87


#define KEY_F1      58
#define KEY_F2      59
#define KEY_F3      60
#define KEY_F4      61
#define KEY_F5      62
#define KEY_F6      63
#define KEY_F7      64
#define KEY_F8      65
#define KEY_F9      66
#define KEY_F10     67
#define KEY_F11     68
#define KEY_F12     69

#define KEY_DELETE  53 // 49,50 = Dividiert

static const uchar  keyReport[NUM_KEYS + 1][2] PROGMEM = {
	/* none */  {0, 0},                     /* no key pressed */
	/*  1 */    {0, KEY_A},
	/*  2 */    {0, KEY_B},
	/*  3 */    {0, KEY_C},
	/*  4 */    {0, KEY_D},
	/*  5 */    {0, KEY_E},
	/*  6 */    {0, KEY_F},
	/*  7 */    {0, KEY_G},
	/*  8 */    {0, KEY_H},
	/*  9 */    {0, KEY_I},
	/* 10 */    {0, KEY_J},
	/* 11 */    {0, KEY_K},
	/* 12 */    {0, KEY_L},
	/* 13 */    {0, KEY_M},
	/* 14 */    {0, KEY_N},
	/* 15 */    {0, KEY_O},
	/* 16 */    {0, KEY_P},
	/* 17 */    {0, KEY_Q},
	/* 18 */    {0, KEY_R},
	/* 19 */    {0, KEY_S},
	/* 20 */    {0, KEY_T},
	/* 21 */    {0, KEY_U},
	/* 22 */    {0, KEY_V},
	/* 23 */    {0, KEY_W},
	/* 24 */    {0, KEY_X},
	/* 25 */    {0, KEY_Y},
	/* 26 */    {0, KEY_Z},
	/* 27 */    {0, KEY_ENTER},
	/* 28 */    {0, KEY_DELETE},
	/* 29 */    {0, KEY_BACKSPACE},
	/* 30 */    {0, KEY_0},
	/* 31 */    {0, KEY_1},
	/* 32 */    {0, KEY_2},
	/* 33 */    {0, KEY_3},
	/* 34 */    {0, KEY_4},
	/* 35 */    {0, KEY_5},
	/* 36 */    {0, KEY_6},
	/* 37 */    {0, KEY_7},
	/* 38 */    {0, KEY_8},
	/* 39 */    {0, KEY_9},
	/* 40 */    {0, KEY_DOT},
	/* 41 */    {0, KEY_SPACE},
	/* 42 */    {KEY_LS, 0},
	/* 43 */    {KEY_RS,0},
	/* 44 */    {0, KEY_ALPHA},
	/* 45 */    {0, KEY_DIV},
	/* 46 */    {0, KEY_MUL},
	/* 47 */    {0, KEY_MINUS},
	/* 48 */    {0, KEY_PLUS},
	/* 49 */    {0, KEY_ON},
};

static void buildReport(uchar key)
{
	/* This (not so elegant) cast saves us 10 bytes of program memory */
	*(int *)reportBuffer = pgm_read_word(keyReport[key]);
}

uchar	usbFunctionSetup(uchar data[8])
{
	usbRequest_t    *rq = (void *)data;

	usbMsgPtr = reportBuffer;
	if((rq->bmRequestType & USBRQ_TYPE_MASK) == USBRQ_TYPE_CLASS){    /* class request type */
		if(rq->bRequest == USBRQ_HID_GET_REPORT){  /* wValue: ReportType (highbyte), ReportID (lowbyte) */
			/* we only have one report type, so don't look at wValue */
			buildReport(keyPressed());
			return sizeof(reportBuffer);
		}else if(rq->bRequest == USBRQ_HID_GET_IDLE){
			usbMsgPtr = &idleRate;
			return 1;
		}else if(rq->bRequest == USBRQ_HID_SET_IDLE){
			idleRate = rq->wValue.bytes[1];
		}
	}else{
		/* no vendor specific requests implemented */
	}
	return 0;
}

/* ------------------------------------------------------------------------- */

int	main(void)
{
	uchar   key, lastKey = 0, keyDidChange = 0;
	uchar   idleCounter = 0;

	wdt_enable(WDTO_2S);
	hardwareInit();
	odDebugInit();
	usbInit();
	sei();
	DBG1(0x00, 0, 0);

	for(;;){	/* main event loop */
		wdt_reset();
		usbPoll();
		key = keyPressed();
		if(lastKey != key){
			lastKey = key;
			keyDidChange = 1;
		}
		if(TIFR & (1<<TOV0)){   /* 22 ms timer */
			TIFR = 1<<TOV0;
			if(idleRate != 0){
				if(idleCounter > 4){
					idleCounter -= 5;   /* 22 ms in units of 4 ms */
				}else{
					idleCounter = idleRate;
					keyDidChange = 1;
				}
			}
		}
		if(keyDidChange && usbInterruptIsReady()){
			keyDidChange = 0;
			/* use last key and not current key status in order to avoid lost
			changes in key status. */
			buildReport(lastKey);
			usbSetInterrupt(reportBuffer, sizeof(reportBuffer));
		}
	}
	return 0;
}

/* ------------------------------------------------------------------------- */
