/*
 * Servo Switch (SESW) - Test Program
 * For more details visit: http://www.oomlout.com/serb 
 * 
 * Behaviour: The default behaviour is to turn the switch on then wait five
 *            seconds before turning the switch off, and waiting five 
 *            seconds. This will repeat. 
 *            There are also different demo modes entered by pulling the
 *            corresponding pin LOW:
 *            pin 3: push button - push a button attached to pin 3 to toggle
 *                                 the state of the switch.
 *            pin 4: proximity - uses an infra red proximity detector to toggle
 *                               the switch when something comes within 20 cm 
 *                               (proximity sense pin attached to analog 0)
 *            pin 5: light - uses a photoresistor to turn the switch on when
 *                           there is not much light and off when there is
 *                           (photo resistor attached to analog 1)
 *           
 * Wiring: Servo Switch Signal - pin 9
 *         Push Button = pulls pin 3 low
 *         Proximity Sensor = Analog 0
 *         Photo Resistor = Analog 1
 *
 * License: This work is licenced under the Creative Commons 
 *          Attribution-Share Alike 3.0 Unported License. To 
 *          view a copy of this licence, visit 
 *          http://creativecommons.org/licenses/by-sa/3.0/ 
 *          or send a letter to Creative Commons, 171 Second 
 *          Street, Suite 300, San Francisco, California 94105, 
 *          USA.
 *         
*/ 


//--------------------------------------------------------------------------
// Servo Switch - (SESW) Preamble START

#include <Servo.h> 
 
#define SERVOSWITCHONEPIN 9   //The pin the first servoSwitch is attached to
Servo servoSwitchOne;         //the Servo object connected to our light switch


//Switch Constants (can be adjusted to fine tune for your switch)
#define SWITCHTIME 500        //the time in milliseconds it takes to flick the switch

boolean switchState;          //the state if the switch
#define ON true               //true when on
#define OFF false             //false when off

#define ONDEGREES  180        //the number of degrees on the servo corresponding to ON
#define NEUTRALDEGREES  75    //the number of degrees on the servo corresponding to NEUTRAL
#define OFFDEGREES  0    //the number of degrees on the servo corresponding to OFF

// Servo Switch - (SESW) Preamble END
//--------------------------------------------------------------------------

//--------------------------------------------------------------------------
// Servo Switch - (SESW) Demo Mode Preamble START

//Demo Mode One Preamble - BUTTON
#define MODEPINBUTTON 3    //pull pin 3 low to go into button mode
#define BUTTONPIN 2        //the pin the button is attached to (active low)


//Demo Mode Two Preamble - PROXIMITY
#define MODEPINPROXIMITY 4 //pull pin 4 low to go to proximity mode
#define PROXIMITYPIN 0     //proximity sense pin goes to analog 0
#define PROXIMITYTHRESHOLD 400 //the value which intiates a toggle (about 20cm) when higher than within 20cm

//Demo Mode Three Preamble - LIGHT
#define MODEPINLIGHT 5 //pull pin 5 low to go to light mode
#define LIGHTPIN 1     //light sense pin goes to analog 1
#define LIGHTTHRESHOLD 305 //the value which intiates a toggle (when shined by a flashlight) ~1volt

// Servo Switch - (SESW) Demo ModePreamble END
//--------------------------------------------------------------------------

long lastRun;
                
/* The setup routine (runs once on startup) */
void setup() 
{ 
  lastRun = millis();           //sets the initial run time
  servoSwitchSetup();           //goes to the servo switch setup routine
  servoSwitchDemoModeSetup();   //sets up the pins used for the various demo modes
} 
 
/* the loop routine will run continuously after setup() finishes */ 
void loop() 
{ 
  boolean options = checkOptions();   //checks if one of the demo option pins is pulled low
  if(!options){                       //if no demo modes are selected
     if(millis() - lastRun > 5000){   //seeing if five seconds have elapsed since we last toggled
     toggle();                        //toggles the state of the switch
     lastRun = millis();              //set lastRun to current time
     }
  }
} 

//--------------------------------------------------------------------------
// Servo Switch - (SESW) Routines START

/* Sets up our servo switch by attaching the servo to the selected pin (must be 9 or 10) */
void servoSwitchSetup(){
  servoSwitchOne.attach(SERVOSWITCHONEPIN);  //Attaches the servo switch's servo 
  turnOn();                                  //turns the switch on to start
}

/* Flicks the switch on then returns to the neutral position */
void turnOn(){
  switchState = ON;                 //strores the current state as on 
  servoSwitchOne.write(ONDEGREES);  //turns the servo to on degrees    
  delay(SWITCHTIME);           //waits for the time it takes to fully flick the switch
  goNeutral();                 //returns the servo to the neutral position
} 

/* Flicks the switch off then returns to the neutral position */
void turnOff(){
  switchState = OFF;                 //strores the current state as off 
  servoSwitchOne.write(OFFDEGREES);  //turns the servo to off degrees    
 delay(SWITCHTIME);           //waits for the time it takes to fully flick the switch
 goNeutral();                 //returns the servo to the neutral position
} 

/* returns the switch to it's neutral position */
void goNeutral(){
 servoSwitchOne.write(NEUTRALDEGREES); //moves the servo to the neutral position
}

/* toggles the state of the switch */
void toggle(){
  if(switchState == ON){turnOff();}
  else{turnOn();}
}

// Servo Switch - (SESW) Routines END
//--------------------------------------------------------------------------


//--------------------------------------------------------------------------
// Servo Switch - (SESW) Demo Mode Routines START

void servoSwitchDemoModeSetup(){
//Demo Mode One Setup - BUTTON
  pinMode(MODEPINBUTTON, INPUT);
  digitalWrite(MODEPINBUTTON, HIGH);
  pinMode(BUTTONPIN, INPUT);
  digitalWrite(BUTTONPIN, HIGH);
//Demo Mode Two Setup - PROXIMITY
  pinMode(MODEPINPROXIMITY, INPUT);
  digitalWrite(MODEPINPROXIMITY, HIGH);
//Demo Mode Three Setup - LIGHT
  pinMode(MODEPINLIGHT, INPUT);
  digitalWrite(MODEPINLIGHT, HIGH);  
}


/* Checks if any of the demo options are active will return false if none are and
 * true if they are (each demo routines code is also in this method)
 */
boolean checkOptions(){
 boolean returnValue = false;    //set the default return value to false
 //Demo Mode 1 - BUTTON
 if(digitalRead(MODEPINBUTTON) == LOW){  //if BUTTON Demo Mode is selected (pin 2 pulled low)
  if(digitalRead(BUTTONPIN) == LOW){     //if button is pressed
    toggle();                            //toggle the state of the switch
    delay(500);                          //wait half a second to avoid double pushing
    lastRun = millis();                  //set lastRun to current time
  }
 returnValue = true;                  //set the return value as true 
 }
 //Demo Mode 2 - PROXIMITY
 if(digitalRead(MODEPINPROXIMITY) == LOW){ //if PROXIMITY Demo Mode is selected (pin 3 pulled low)
  if(analogRead(PROXIMITYPIN) > PROXIMITYTHRESHOLD){ //if object is closer than the threchold distance
    toggle();            //toggle the state of the switch
    delay(1000);         //wait a second to avoid double toggling
    lastRun = millis();  //set lastRun to current time
  }
 returnValue = true;  //sets the return value as true
 }
//Demo Mode 3 - LIGHT
 if(digitalRead(MODEPINLIGHT) == LOW){ //if Light Demo Mode is selected (pin 4 pulled low)
  if(analogRead(LIGHTPIN) < LIGHTTHRESHOLD){ //if light is brighter than the threshold value
    toggle();            //toggle the state of the switch
    delay(1000);         //wait a second to avoid double toggling
    lastRun = millis();  //set lastRun to current time
  }  
  returnValue = true;
 }
 return returnValue;
}

// Servo Switch - (SESW) Demo Mode Routines END
//--------------------------------------------------------------------------

